/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim: set ts=8 sts=2 et sw=2 tw=80: */
/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this file,
 * You can obtain one at http://mozilla.org/MPL/2.0/. */

#ifndef DOM_NOTIFICATION_NOTIFICATIONPARENT_H_
#define DOM_NOTIFICATION_NOTIFICATIONPARENT_H_

#include "mozilla/dom/notification/PNotificationParent.h"
#include "mozilla/ipc/PBackgroundParent.h"
#include "mozilla/dom/DOMTypes.h"

namespace mozilla::dom::notification {

enum class CloseMode;

enum class AlertTopic : uint8_t {
  Disable,
  Settings,
  Click,
  Show,
  Finished,
};

struct NotificationParentArgs {
  NotNull<nsCOMPtr<nsIPrincipal>> mPrincipal;
  NotNull<nsCOMPtr<nsIPrincipal>> mEffectiveStoragePrincipal;
  bool mIsSecureContext;
  nsString mScope;
  IPCNotification mNotification;
};

class NotificationParent final : public PNotificationParent,
                                 public nsISupports,
                                 public SupportsWeakPtr {
  using IPCResult = mozilla::ipc::IPCResult;

 public:
  NS_DECL_ISUPPORTS

  nsresult HandleAlertTopic(AlertTopic aTopic);
  IPCResult RecvShow(ShowResolver&& aResolver);
  IPCResult RecvClose();

  static nsresult CreateOnMainThread(
      NotificationParentArgs&& mArgs,
      Endpoint<PNotificationParent>&& aParentEndpoint,
      PBackgroundParent::CreateNotificationParentResolver&& aResolver);

 private:
  explicit NotificationParent(NotificationParentArgs&& aArgs)
      : mId(aArgs.mNotification.id()), mArgs(std::move(aArgs)) {};
  ~NotificationParent() = default;

  nsresult Show();
  nsresult FireClickEvent();

  void Unregister();

  Maybe<NotificationParent::ShowResolver> mResolver;

  // The ID generated by nsIAlertNotification during Show() function. It may
  // stay empty if the function fails.
  nsString mId;
  const NotificationParentArgs mArgs;

  // Whether it's now a dangling actor without corresponding OS notification,
  // either because it's closed or denied permission. We don't have to call
  // CloseAlert if this is the case.
  bool mDangling = false;
};

}  // namespace mozilla::dom::notification

#endif  // DOM_NOTIFICATION_NOTIFICATIONPARENT_H_
